<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use PhonePe\PhonePe;
use Illuminate\Support\Facades\Log;
use Auth;

class PhonePayController extends Controller
{
    public function initiatePayment(Request $request)
    {
        $merchantId = env('PHONEPE_MERCHANT_ID');
        $apiKey = env('PHONEPE_SALT_KEY');
        $redirect = env('PHONEPE_REDIRECT_URL');

        // Prepare the transaction data
        $transactionData = [
            'merchantId' => $merchantId,
            'merchantTransactionId' => 'MT' . uniqid(),
            'merchantUserId' => $merchantId,
            'amount' => $request->amount * 100,
            'redirectUrl' => $redirect . '/redirect-url',
            'redirectMode' => 'REDIRECT',
            'callbackUrl' => $redirect . '/callback-url',
            'mobileNumber' => Auth::user()->mobile,
            'paymentInstrument' => [
                'type' => 'PAY_PAGE',
            ],
        ];

        // Encode and prepare headers
        $payloadMain = base64_encode(json_encode($transactionData));
        $saltIndex = 1;
        $payload = $payloadMain . "/pg/v1/pay" . $apiKey;
        $sha256 = hash("sha256", $payload);
        $finalXHeader = $sha256 . '###' . $saltIndex;

        try {
            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'X-VERIFY' => $finalXHeader,
                'accept' => 'application/json',
            ])->post('https://api.phonepe.com/apis/hermes/pg/v1/pay', [
                'request' => $payloadMain
            ]);

            if ($response->successful()) {
                $data = $response->json();

                if (isset($data['code']) && $data['code'] === 'PAYMENT_INITIATED') {
                    $payUrl = $data['data']['instrumentResponse']['redirectInfo']['url'] ?? null;
                    if ($payUrl) {
                        return redirect()->away($payUrl);
                    } else {
                        return back()->withErrors(['error' => 'Payment URL not found in response']);
                    }
                } else {
                    Log::error('Payment initiation failed', ['response' => $data]);
                    return back()->withErrors(['error' => 'Payment initiation failed']);
                }
            } else {
                Log::error('Payment request failed', [
                    'status' => $response->status(),
                    'body' => $response->json()
                ]);

                return back()->withErrors(['error' => 'Payment request failed with status: ']);
            }
        } catch (\Exception $e) {
            Log::error('Payment API Error: ' . $e->getMessage());
            return back()->withErrors(['error' => 'An unexpected error occurred. Please try again later.']);
        }
    }

    // Updated handler for redirect URL
    public function handleRedirect(Request $request)
    {
        try {
            $merchantTransactionId = $request->input('merchantTransactionId');
            $transactionId = $request->input('transactionId');
            $status = $request->input('status');

            if ($status === 'SUCCESS') {
                // Set payment details
                $payment_id = $transactionId;
                $payment_amount = $request->amount * 100;
                $payment_method = 'phonepe';
                $payment_status = 1;
                $plan_id = $request->plan_id;

                // Initialize SubscriptionController
                $checkout = new SubscriptionController;
                $checkout->subscribe($payment_id, $payment_method, $plan_id, $payment_status, $payment_amount);

                // Redirect to home with success message
                return redirect('/')->with('success', 'Payment completed successfully!');
            } else {
                return redirect('/')->with('error', 'Payment failed. Please try again.');
            }
        } catch (\Exception $e) {
            Log::error('Redirect handling error: ' . $e->getMessage());
            return redirect('/')->with('error', 'Error processing payment. Please contact support.');
        }
    }

    // Updated handler for callback URL
    public function handleCallback(Request $request)
    {
        try {
            Log::info('PhonePe Callback received', $request->all());
            
            $status = $request->input('status');
            
            if ($status === 'SUCCESS') {
                return redirect('/')->with('success', 'Payment completed successfully!');
            } else {
                return redirect('/')->with('error', 'Payment failed. Please try again.');
            }
        } catch (\Exception $e) {
            Log::error('Callback handling error: ' . $e->getMessage());
            return redirect('/')->with('error', 'Error processing payment. Please contact support.');
        }
    }
}